//*************************************************************************************************
//
//	Description:
//		basic.fx - Basic object shader for The Race. Supports vertex colours, Blinn-style diffuse
//		and specular lighting, shininess maps, tangent space normal maps, multiple light sources
//		(of 3 different types), shadow buffers.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/05/2006  0.1           Created
//		TNettleship     05/12/2006  1.0           First "offical" release to FTP
//		TNettleship     05/18/2006  1.1           Added technique annotations.
//		TNettleship     05/31/2006  1.2           Added renderer lighting interface.
//		TNettleship     05/31/2006  1.3           Added specular overlighting to game engine code.
//		TNettleship     06/08/2006  1.4           Added ambient occlusion term from vertex colour,
//																							global emap factor, added option for compressed
//																							normals.
//		TNettleship			08/01/2006	1.5						Added option for ambient occlusion to affect
//																							diffuse and specular lighting. Fixed a bug with
//																							channel assignment & vertex colours in 3DSMAX.
//		TNettleship			10/16/2006	1.6						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.7						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			11/02/2006	1.8						Fixed a bug with directional lighting.
//		TNettleship			11/21/2006	1.9						Added support for deferred rendering.
//		TMann						11/23/2006	1.12					Added _WIN_GL_ support
//		TMann						11/27/2006	1.13					Texture filtering for GL
//		TNettleship			12/04/2006	1.14					Added premultiplied alpha code to the fragment shader,
//		TMann						12/11/2006	1.15					PS3 support
//																							added support for double sided rendering.
//		TNettleship			01/02/2007	1.16					Stopped premultiplied alpha from being used in opaque
//																							materials.
//		TMann						01/09/2007	1.17					Added PS3/GL literal param passing
//		TNettleship			01/31/2007	1.18					Pixel shader optimisations
//		TNettleship			02/06/2007	1.19					Fixed translucent_twosided technique
//		TNettleship			02/12/2007	1.20					Fixed environment mapping
//		BIrvine					02/13/2007	1.21					Temporary shadow implementation
//		TNettleship			04/18/2007	2.0						Added shader specialisation.
//		TNettleship			05/03/2007	2.01					Added shadow override technique for basic_translucent.
//		TNettleship			07/10/2007	2.02					Added support for vertex alpha, added alphatest technique.
//		TNettleship			07/10/2007	2.03					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.04          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.05          Removed texture bias.
//		TNettleship			08/30/2007	2.06					Mods for atoc support.
//		TNettleship			09/25/2007	2.07					Removed obsolete techniques, changed fresnel to be an
//																							optional specialisation.
//		TNettleship			10/23/2007	2.08					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f

// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define USE_ANISO
#define USE_SPECULAR
#define NORMAL_MAPPING
#define ENV_MAPPING
#define USE_FRESNEL
#endif


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

SPECIALISATION_PARAM( useAnisotropic, "Anisotropic Filter?", "USE_ANISO" )	// TRUE if the anisotropic filtering is to be used

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

SPECIALISATION_PARAM_DEFAULTS_TRUE( useSpecular, "Use specular?", "USE_SPECULAR" )	// TRUE if the specular lighting is to be used

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_TEXTURE_PARAM( specularTexture, "Spec Tex {UV1}", useSpecular )
#endif

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
DEPENDENT_BOOL_PARAM( specialNormalMap, "Special normal map format?", useNormalMap )
#endif

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

SPECIALISATION_PARAM( useFresnel, "Use Fresnel?", "USE_FRESNEL" )	// TRUE if fresnel factor should be used

#if defined( _3DSMAX_ ) || defined( USE_FRESNEL )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( fresnelFactor, "Fresnel Factor", useFresnel, 0.0f, 1.0f, 0.6f )
#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( minSpecPower, "Min Specular Power", useSpecular, 1.0f, 1024.0f, 1.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( maxSpecPower, "Max Specular Power", useSpecular, 1.0f, 1024.0f, 32.0f )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalSpecularFactor, "Specular Factor", useSpecular, 0.0f, 10.0f, 1.0f )
#endif

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 1.0f, 1.0f )
#endif



//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

//
// Following parameters are read by the game engine, the shader does not read them directly
//

// the time each transition should take
float frameTransitionTime 
<
	string UIName = "Ad Transition Time";
	bool appEdit = true;
	bool export = true;
> = 1.0;

// the time ad frame should be displayed for
float frameDisplayTime
<
	string UIName = "Ad Display Time";
	bool appEdit = true;
	bool export = true;
> = 5.0;

// use this param if you want to have ads that are not in sync with each other
float frameInitialTimeOffset
<
	string UIName = "Ad Time Offset";
	bool appEdit = true;
	bool export = true;
> = 0.0;

int frameDimX
<
	string UIName = "Sub Texture Columns";
	bool appEdit = true;
	bool export = true;
> = 4;

int frameDimY
<
	string UIName = "Sub Texture Rows";
	bool appEdit = true;
	bool export = true;
> = 4;



// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };

//
// animated frames
//

float4 frameA 
<
	string UIWidget = "None";
	bool appEdit = true;
> = float4(0, 0, 0.25, 0.25);

float4 frameB
<
	string UIWidget = "None";
	bool appEdit = true;
> = float4(0.25, 0.25, 0.5, 0.5);

// value is between 0 .. 1, the game animates this value 
float frameTransition
<
	string UIWidget = "None";
	bool appEdit = true;
> = 0;

//-----------------------------------------------------------------------
//
// Samplers
//

#if defined( USE_ANISO )

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

#else

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#endif

#if defined( _3DSMAX_ ) || defined( USE_SPECULAR )
sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif



//
// Functions
//


//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float CalculateFresnel( float3 _eye, float3 _normal, float rindexRatio )
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
	float3 normal   : NORMAL;															// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent  : TANGENT;														// Object space tangent
	float3 binormal : BINORMAL;														// Object space normal
#endif
};


struct SHADOWGEN_VSINPUT
{
	float3 position : POSITION;														// Object space position
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// Texture coords
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};



struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 coords			: TEXCOORD1;
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT BasicVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	return _output;
}



SHADOWGEN_VSOUTPUT BasicTranslucentShadowGenVertexShader( SHADOWGEN_VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

	OUTPUT_SHADOW_COORDS;

	return _output;
}


ZPRIMEDOF_VSOUTPUT BasicTranslucentZPrimeDOFVertexShader( SHADOWGEN_VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;
	_output.colour	 = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD1;											// Vertex colour
	float2 texCoord		: TEXCOORD0;									// UV coords for texture channel 0
};


struct SHADOWGEN_PSINPUT
{
	float4 colour				: TEXCOORD2;
	float2 texCoord			: TEXCOORD0;
	float4 shadowCoord	: TEXCOORD1;
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

float4 FrameTransition( float4 fA, float4 fB, float frameTransition )
{
	return lerp( fA, fB, frameTransition );
}


float4 TexFrames( sampler2D s, float2 uv )
{
	float4 fA = tex2D( s, lerp( frameA.xy, frameA.zw, uv) );
	float4 fB = tex2D( s, lerp( frameB.xy, frameB.zw, uv) );
	return FrameTransition( fA, fB, frameTransition );

/*	int frameV = frame / 4;
	int frameU = frame - 4*frameV;
	
	return tex2D( s, 0.25 * uv + float2( frameU * 0.25, frameV * 0.25 ) );*/
}

REMOVE_UNUSED_INTERPOLATORS	// NOTE: this is commented out because the Vertex Shader is giving outputs that the pixel shader doesn't need, but I can't figure out the combination of defines to fix this quickly!!
PSOUTPUT BasicFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = TexFrames( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	_output.Colour = CalculateOutputPixel( diffuseTexColour );

	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};


VSOUTPUT_LD BasicLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = TexFrames( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	_output.Colour = CalculateLowDetailOutputPixel( diffuseTexColour );

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicTranslucentShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = TexFrames( diffuseMap, _input.texCoord );

	CALC_SHADOWMAP_DEPTH( output.Colour, _input.shadowCoord );
	output.Colour.a = diffuseTexColour.a * _input.colour.a;
	SHADOWMAP_PS_ALPHATEST( output.Colour.a, 0.25f );

	return output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT BasicTranslucentZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT output;

	float4 diffuseTexColour = TexFrames( diffuseMap, _input.texCoord );

	output.Colour = _input.coords.z / _input.coords.w;
	output.Colour.a = diffuseTexColour.a * _input.colour.a;

	return output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Basic
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Basic_LowDetail";
	int    lowDetailDeferredID = 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicVertexShader();
		PixelShader = compile sce_fp_rsx BasicFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique Basic_Translucent
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "Basic_LowDetail";
	int    lowDetailDeferredID	= 2;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



//
// Two sided technique is now obsolete
//
technique Basic_TwoSided
<
	bool obsolete = true;
	string replacementTechnique = "Basic";
>
{
#ifdef _3DSMAX_
	// Even though this is an obsolete technique, 3dsmax parser 0x0001 will crash
	// if it doesn't have a VS and PS defined.
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}


//
// Two sided translucent technique is now obsolete
//
technique Basic_Translucent_TwoSided
<
	bool obsolete = true;
	string replacementTechnique = "Basic_Translucent";
>
{
#ifdef _3DSMAX_
	// Even though this is an obsolete technique, 3dsmax parser 0x0001 will crash
	// if it doesn't have a VS and PS defined.
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}



technique Basic_AlphaTest
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_Basic_Translucent_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
	string lowDetailBehaviour		= "ERMB_RENDER";
	string lowDetailTechnique		= "Basic_LowDetail";
	int    lowDetailDeferredID	= 0;
	bool   appCanOverrideSampler = true;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = true;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif


#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique Basic_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Basic_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx BasicLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx BasicLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 BasicLowDetailVertexShader();
		PixelShader = compile ps_3_0 BasicLowDetailFragmentShader();
#endif
	}
}


technique Basic_Translucent_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic_LowDetail";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_DONT_RENDER";
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "INVSRCALPHA";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = INVSRCALPHA;
		BlendOp = ADD;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}


//
// Two sided technique is now obsolete
//
technique Basic_TwoSided_LowDetail
<
	bool obsolete = true;
	string replacementTechnique = "Basic_LowDetail";
>
{
#ifdef _3DSMAX_
	// Even though this is an obsolete technique, 3dsmax parser 0x0001 will crash
	// if it doesn't have a VS and PS defined.
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}


//
// Two sided translucent technique is now obsolete
//
technique Basic_Translucent_TwoSided_LowDetail
<
	bool obsolete = true;
	string replacementTechnique = "Basic_Translucent_LowDetail";
>
{
#ifdef _3DSMAX_
	// Even though this is an obsolete technique, 3dsmax parser 0x0001 will crash
	// if it doesn't have a VS and PS defined.
	pass Pass0
	{
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
	}
#endif
}


//
// This technique only differs from Basic by the renderstates
// it requires, so its a max-only technique.
//
technique Basic_AlphaTest_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "Basic_LowDetail";
	int    normalDeferredID			= 0;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_Basic_Translucent_ZPrime_DOF";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_Basic_Translucent_ShadowGen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = true;
		bool	AlphaBlendEnable = false;
		bool	AlphaTestEnable = true;
		int		AlphaRef = 64;
		string AlphaFunc = "GreaterEqual";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
#endif

#ifdef _3DSMAX_
		// The rendermode mapping table above maps rendering of this technique
		// to other techniques in all modes, so it doesn't need its own compiled
		// shaders, except in max.
		VertexShader = compile vs_3_0 BasicVertexShader();
		PixelShader = compile ps_3_0 BasicFragmentShader();
#endif
	}
}



technique _Basic_Translucent_ShadowGen
{
	pass Pass0
	{
			SHADOWMAP_STATES_ALPHATEST( 0x40 )
#if defined (_PS3_)
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicTranslucentShadowGenVertexShader();
			PixelShader = compile sce_fp_rsx BasicTranslucentShadowGenFragmentShader();
#elif defined (_XBOX)
			VertexShader = compile vs_3_0 BasicTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicTranslucentShadowGenFragmentShader();
#else
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicTranslucentShadowGenVertexShader();
			PixelShader = compile ps_3_0 BasicTranslucentShadowGenFragmentShader();
#endif
	}
}

technique _Basic_Translucent_ZPrime_DOF
{
	pass Pass0
	{
#if defined (_PS3_)
			AlphaTestEnable = true;
		  AlphaFunc = int2(GEqual, 0x40);
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile sce_vp_rsx BasicTranslucentZPrimeDOFVertexShader();
			PixelShader = compile sce_fp_rsx BasicTranslucentZPrimeDOFFragmentShader();
#else
			AlphaTestEnable = true;
			AlphaRef = 0x40;
		  AlphaFunc = GreaterEqual;
			ZEnable = true;
			ZWriteEnable = true;
			VertexShader = compile vs_3_0 BasicTranslucentZPrimeDOFVertexShader();
			PixelShader = compile ps_3_0 BasicTranslucentZPrimeDOFFragmentShader();
#endif
	}
}
